﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace DocumentProcessing
{
    // Listing 5-32. Własny sposób przechowywania informacji o zdarzeniach
    class DocumentProcessor
    {
        private Dictionary<string, Delegate> events;
        public event EventHandler<ProcessCancelEventArgs> Processing
        {
            add
            {
                Delegate theDelegate =
                EnsureEvent("Processing");
                events["Processing"] =
                ((EventHandler<ProcessCancelEventArgs>)
                theDelegate) + value;
            }
            remove
            {
                Delegate theDelegate =
                EnsureEvent("Processing");
                events["Processing"] =
                ((EventHandler<ProcessCancelEventArgs>)
                theDelegate) - value;
            }
        }
        public event EventHandler<ProcessEventArgs> Processed
        {
            add
            {
                Delegate theDelegate =
                EnsureEvent("Processed");
                events["Processed"] =
                ((EventHandler<ProcessEventArgs>)
                theDelegate) + value;
            }
            remove
            {
                Delegate theDelegate =
                EnsureEvent("Processed");
                events["Processed"] =
                ((EventHandler<ProcessEventArgs>)
                theDelegate) - value;
            }
        }
        private Delegate EnsureEvent(string eventName)
        {
            // Tworzymy słownik, jeśli jeszcze nie istnieje.
            if (events == null)
            {
                events = new Dictionary<string, Delegate>();
            }
            // Dodajemy puste miejsce na delegację, jeśli jeszcze
            // go nie mamy.
            Delegate theDelegate = null;
            if (!events.TryGetValue(
            eventName, out theDelegate))
            {
                events.Add(eventName, null);
            }
            return theDelegate;
        }
        private void OnProcessing(ProcessCancelEventArgs e)
        {
            Delegate eh = null;
            if (events != null &&
            events.TryGetValue("Processing", out eh))
            {
                EventHandler<ProcessCancelEventArgs> pceh =
                eh as EventHandler<ProcessCancelEventArgs>;
                if (pceh != null)
                {
                    pceh(this, e);
                }
            }
        }
        private void OnProcessed(ProcessEventArgs e)
        {
            Delegate eh = null;
            if (events != null &&
            events.TryGetValue("Processed", out eh))
            {
                EventHandler<ProcessEventArgs> pceh =
                eh as EventHandler<ProcessEventArgs>;
                if (pceh != null)
                {
                    pceh(this, e);
                }
            }
        }

        public Func<Document, string> LogTextProvider
        {
            get;
            set;
        }


        class ActionCheckPair
        {
            public Action<Document> Action { get; set; }
            public Predicate<Document> QuickCheck { get; set; }
        }

        private readonly List<ActionCheckPair> processes =
            new List<ActionCheckPair>();

        public void AddProcess(Action<Document> action)
        {
            AddProcess(action, null);
        }

        public void AddProcess(Action<Document> action,
            Predicate<Document> quickCheck)
        {
            processes.Add(
            new ActionCheckPair { Action = action, QuickCheck = quickCheck });
        }

        public void Process(Document doc)
        {
            ProcessEventArgs e = new ProcessEventArgs(doc);
            ProcessCancelEventArgs ce = new ProcessCancelEventArgs(doc);
            OnProcessing(ce);
            if (ce.Cancel)
            {
                Console.WriteLine("Proces został anulowany.");
                if (LogTextProvider != null)
                {
                    Console.WriteLine(LogTextProvider(doc));
                }
                return;
            }

            // Najpierw przeprowadzamy szybką weryfikację dokumentu.
            foreach (ActionCheckPair process in processes)
            {
                if (process.QuickCheck != null && !process.QuickCheck(doc))
                {
                    Console.WriteLine("Przetwarzanie nie zakończy się pomyślnie.");
                    if (LogTextProvider != null)
                    {
                        Console.WriteLine(LogTextProvider(doc));
                    }
                    OnProcessed(e);
                    return;
                }
            }
            // Teraz wykonujemy akcję
            foreach (ActionCheckPair process in processes)
            {
                process.Action(doc);
                if (LogTextProvider != null)
                {
                    Console.WriteLine(LogTextProvider(doc));
                }
            }
            OnProcessed(e);
        }
    }
}